/*
 * @(#)OMBidirectionalLink.java  1.0  4. Februar 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.oo.objectmodel;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.beans.*;
import java.util.*;
/**
 * Simulates the structure and behavior of a bidirectional link between two
 * objects.
 *
 * @author  Werner Randelshofer
 * @version 1.0 4. Februar 2004  Created.
 */
public class OMBidirectionalLink extends OMLink {
    private String ownerLabel = "owner";
    private boolean isOwnerTraversable = false;
    
    /** Creates a new instance. */
    public OMBidirectionalLink() {
        changeSupport = new PropertyChangeSupport(this);
    }
    
    public void setAssociation(OMAssociation newValue) {
        super.setAssociation(newValue);
        
        // FIXME - We should fire events when newValue = null as well.
        if (newValue != null) {
            changeSupport.firePropertyChange("ownerLabel", ownerLabel, newValue.getOwnerLabel());
        }
    }
    
    public void addPropertyChangeListener(PropertyChangeListener l) {
        changeSupport.addPropertyChangeListener(l);
    }
    public void removePropertyChangeListener(PropertyChangeListener l) {
        changeSupport.removePropertyChangeListener(l);
    }
    
    public OMObject getAssociate(OMObject startOrEnd) {
        return (OMObject) getConnected(startOrEnd);
    }
    
    public void setOwnerLabel(String newValue) {
        String oldValue = ownerLabel;
        ownerLabel = newValue;
        changeSupport.firePropertyChange("ownerLabel", oldValue, newValue);
    }
    public String getOwnerLabel() {
        return (getAssociation() == null) ? ownerLabel : getAssociation().getOwnerLabel();
    }
    
    public void setOwnerTraversable(boolean newValue) {
        boolean oldValue = isOwnerTraversable;
        isOwnerTraversable = newValue;
        changeSupport.firePropertyChange("ownerTraversable", oldValue, newValue);
    }
    public boolean isOwnerTraversable() {
        return (getAssociation() == null) ? isOwnerTraversable : getAssociation().isOwnerTraversable();
    }
    
    public int getSimulatedConcept() {
        return ObjectModel.LINK;
    }
    public OMBidirectionalLink clone() {
        OMBidirectionalLink that = (OMBidirectionalLink) super.clone();
        return that;
    }
    
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        int concept1 = start.getSimulatedConcept();
        int concept2 = end.getSimulatedConcept();
        
        if (concept1 == ObjectModel.CLASS && concept2 == ObjectModel.CLASS
        || concept1 == ObjectModel.CLASS && concept2 == ObjectModel.OBJECT
        || concept1 == ObjectModel.OBJECT && concept2 == ObjectModel.CLASS) {
            return true;
        } else if (concept1 == ObjectModel.OBJECT && concept2 == ObjectModel.OBJECT) {
            OMObject o1 = (OMObject) start;
            OMObject o2 = (OMObject) end;
            if (o1.getSimulatedClass() == null || o2.getSimulatedClass() == null) {
                return true;
            } else {
                return findAssociationFor(o1, o2) != null;
            }
        }
        return false;
    }
    
    public void write(DOMOutput out) throws IOException {
        out.openElement("start");
        out.addAttribute("name", ownerLabel);
        if (isOwnerTraversable) {
            out.addAttribute("traversable", isOwnerTraversable);
        }
        out.writeObject(getStart());
        out.closeElement();
        
        out.openElement("end");
        out.addAttribute("name", memberLabel);
        if (isMemberTraversable) {
            out.addAttribute("traversable", isMemberTraversable);
        }
        out.writeObject(getEnd());
        out.closeElement();
        
        out.openElement("association");
        out.writeObject(association);
        out.closeElement();
    }
    public void read(DOMInput in) throws IOException {
        in.openElement("start");
        ownerLabel = in.getAttribute("name", "owner");
        isOwnerTraversable = in.getAttribute("traversable", false);
        setStart((SimulatedElement) in.readObject());
        in.closeElement();
        
        in.openElement("end");
        memberLabel = in.getAttribute("name", "member");
        isMemberTraversable = in.getAttribute("traversable", false);
        setEnd((SimulatedElement)  in.readObject());
        in.closeElement();
        
        in.openElement("association");
       setAssociation((OMAssociation) in.readObject());
        in.closeElement();
    }
}
